function ModelViewer(varargin)
% Dr. A. I. Hanna (2005) CMP, UEA, 2005.
error(nargchk(0,inf,nargin));
ad.options.pdm = '';
ad.options.sfam = '';
ad.options.PMdirectory = '';
ad.pdm = [];
ad.edges = [];
ad.pmpts = [];
if mod(length(varargin),2) ~= 0
    % input args have not com in pairs, woe is me
    error('Arguments to ModelViewer must come param/value in pairs.')
end
for i=1:2:length(varargin)
    switch lower(varargin{i})
        case 'pdmfile'
            ad.options.pdm = varargin{i+1};
        case 'sfamfile'
            ad.options.sfam = varargin{i+1};
        case 'pmdirectory'
            ad.options.PMdirectory = varargin{i+1};
        case 'edges'
            ad.edges = varargin{i+1};
        otherwise
            error(['Unknown parameter name passed to ModelViewer.  Name was ' varargin{i}])
    end
end
if isempty(ad.options)
    error('Options parameter is required.')
end
% Open, move, and get the handles from the figure file.
fig = openfig(mfilename, 'reuse');
% Move the gui and then show it, rather than have it jump all over the
% place.
movegui(fig, 'center');
set(fig, 'visible', 'on');
handles = guihandles(fig);
handles = SetGUILevel(handles, getGUILevelInfo);

set(handles.modelNameTxt, 'String', [getCurrentAAMProjectName, filesep, ad.options.pdm]);
set(fig, 'Color', get(handles.modelPanel, 'BackgroundColor'))
handles.app_modes_val = [];
handles.shape_modes_val = [];

% Set all the callback functions
set(handles.shape_pc_slider, 'callback', {@doUpdateShapeMode});
set(handles.app_pc_slider, 'callback', {@doUpdateAppMode});

% The shape controls
set(handles.walk_curr_shape_pc_btn, 'callback', {@doWalkCurrentShapePC});
set(handles.walk_all_shape_pc_btn, 'callback', {@doWalkAllShapePC});
set(handles.reset_shape_btn, 'callback', {@doResetShape});
set(handles.reset_all_shape_btn, 'callback', {@doResetAllShape});
set(handles.shapeslider, 'callback', {@doSlideModes, handles.shapelevels});
set(handles.shapelevels, 'ButtonDownFcn', {@doSelectMode, handles.shapeslider, 'shape'});

% The appearance controls
set(handles.walk_curr_app_pc_btn, 'callback', {@doWalkCurrentAppPC});
set(handles.walk_all_app_pc_btn, 'callback', {@doWalkAllAppPC});
set(handles.reset_app_btn, 'callback', {@doResetApp});
set(handles.reset_all_app_btn, 'callback', {@doResetAllApp});
set(handles.appslider, 'callback', {@doSlideModes, handles.applevels});
set(handles.applevels, 'ButtonDownFcn', {@doSelectMode, handles.appslider, 'app'});

% Handles to the rest of the GUI
set([handles.orient_ij_rad, handles.orient_xy_rad], 'callback', {@doUpdateDisplay});
set([handles.edges_chk, handles.show_landmarks_chk, handles.show_app_model_chk, handles.tps_chk], 'callback', {@doUpdateDisplay});
set(handles.save_image_btn, 'callback', {@doSaveImage});
set(handles.load_pm_file_btn, 'callback', {@doLoadPointModelFile});
set(handles.calc_weights_btn, 'callback', {@doCalcWeights});
set(handles.import_mean_shape_btn, 'callback', {@doImportMeanShape});
set(handles.add_rand_axis_btn, 'callback', {@doAddRandomShapeAxis});
set(handles.reload_model_btn, 'callback', {@doReloadModel});
set([handles.shape_range, handles.app_range], 'callback', {@doUpdateWalkerLimits});
set([handles.cart_axes_chk], 'callback', {@doChangeAxes});



% Initialize the application data structure
ad.figMain = fig;
ad.handles = handles;
ad.app_levels_plot_handle = [];
ad.currentShapeMode = [];
ad.currentAppMode = [];
ad.currentImage = [];
[ad.segmentPoint, ad.offsets] = isAugmentedProject;
ad.dy= 15;
ad.shapewalkerlim = 3;
ad.appwalkerlim = 3;
ad = loadModels(ad);
setappdata(0,'ModelViewerData',ad);
initgui;
try
    uiwait(fig);
catch
    if ishandle(fig)
        delete(fig)
    end
end
if isappdata(0,'ModelViewerData')
    ad = getappdata(0,'ModelViewerData');
    rmappdata(0,'ModelViewerData')
else
    % figure was deleted
    selection = [];
    value = 0;
end
return;
%%%%%
%
%
%%%%%
function [segmentpoint, offsets] = isAugmentedProject
segmentpoint = [];
segmentpointfile = [pwd, filesep, 'Data', filesep, 'segmentpoint.mat'];
if exist(segmentpointfile)
    segmentpoint = load(segmentpointfile);
    segmentpoint = segmentpoint.segmentpoint;
end
offsetfile = [pwd, filesep, 'Data', filesep, 'offsetinfo.mat'];
if exist(offsetfile)
    offsets = load(offsetfile);
    offsets = offsets.offset;
else
    offsets = zeros(length(segmentpoint)+1, 3);
end

%%%%%
%
%
%%%%%
function doUpdateDisplay(axis_radio, evd)
ad = getappdata(0,'ModelViewerData');
set(ad.figMain, 'Pointer', 'watch');
drawnow;
plotModel;
title(ad.handles.mainAxes, '', 'Interpreter', 'none');

set(ad.figMain, 'Pointer', 'arrow');
return;
%%%%%
%
%
%%%%%
function ad = loadModels(ad)
opt = ad.options;
% Load the pdm if it exists
pdmfile = opt.pdm;
ad.pdmfilename = [pdmfile];
if exist(ad.pdmfilename)
    ad.pdm = load(ad.pdmfilename);
    ad.pdm = ad.pdm.pdm;
    if get(ad.handles.cart_axes_chk, 'Value')
        ad.pdm.P = eye(size(ad.pdm.P,1));
        ad.pdm.b = ones(size(ad.pdm.P,1), 1);
    end
    ad.handles.shape_modes_val = zeros(length(ad.pdm.b),1);
    ad.currentShapeModel = getCurrShapeModel(ad.pdm,  ad.handles.shape_modes_val);
end

% Load the sfam if it exists
sfamfile = opt.sfam;
if ~isempty(sfamfile)
    ad.sfamfilename = [sfamfile];
    set(ad.handles.no_app_txt, 'Visible', 'off');
    if exist(ad.sfamfilename)
        ad.sfam = load(ad.sfamfilename);
        ad.sfam = ad.sfam.sfam;
        ad.handles.app_modes_val = zeros(length(ad.sfam.b),1);
    end
else
    ad.handles = toggleAppearanceGUI(ad.handles, 'off');
end
ad = setupSliderLimits(ad);
return;
%%%%%
%
%
%%%%%
function ad = setupSliderLimits(ad)
app_y_lim = length(ad.handles.app_modes_val) - ad.dy;
shape_y_lim = length(ad.handles.shape_modes_val) - ad.dy;
if -app_y_lim>=-1
    set(ad.handles.appslider, 'Enable', 'off');
else
    set(ad.handles.appslider, 'Min', -app_y_lim-1);
end
if -shape_y_lim>=-1
    set(ad.handles.shapeslider, 'Enable', 'off');
else
    set(ad.handles.shapeslider, 'Min', -shape_y_lim-1);
end
ad.currentShapeMode = 1;
ad.currentAppMode = 1;
return;
%%%%%
%
%
%%%%%
function togglePlotView
ad =getappdata(0, 'ModelViewerData');
if get(ad.handles.orient_xy_rad, 'Value');
    axis(ad.handles.mainAxes, 'xy');
else
    axis(ad.handles.mainAxes, 'ij');
end
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function ad = updateAxisLimits(ad, pts)
xrange = [min(pts(:,1)), max(pts(:,1))];
yrange = [min(pts(:,2)), max(pts(:,2))];
if xrange(1)<ad.xrange(1); ad.xrange(1) = xrange(1); end;
if xrange(2)>ad.xrange(2); ad.xrange(2) = xrange(2); end;
if yrange(1)<ad.yrange(1); ad.yrange(1) = yrange(1); end;
if yrange(2)>ad.yrange(2); ad.yrange(2) = yrange(2); end;
return;
%%%%%
%
%
%%%%%
function initgui
ad =getappdata(0, 'ModelViewerData');
if isempty(ad.pdm)
    uiwait(msgbox('Could not find a shape model. Aborting.','No PDM.','modal'));
    close(ad.figMain);
    return;
end
Xm = ad.pdm.Xm;
mean_pts = reshape(Xm, 2, length(Xm)/2)';
ad.xrange = [min(mean_pts(:,1)), max(mean_pts(:,1))];
ad.yrange = [min(mean_pts(:,2)), max(mean_pts(:,2))];
displayModes(ad.handles.shapelevels, ad.handles.shape_modes_val, ad.handles.shapeslider, 'shape');
ad = updateShapeWalkerLimits(ad);
if ~isempty(ad.options.sfam)
    displayModes(ad.handles.applevels, ad.handles.app_modes_val, ad.handles.appslider, 'app');
    ad = updateAppWalkerLimits(ad);
end
setappdata(0,'ModelViewerData',ad);
plotModel;
ad =getappdata(0, 'ModelViewerData');
axis(ad.handles.mainAxes, 'tight');
ad.xrange = get(ad.handles.mainAxes, 'Xlim');
ad.yrange = get(ad.handles.mainAxes, 'Ylim');
setappdata(0,'ModelViewerData',ad);

%%%%%
%
%
%%%%%
function doChangeAxes(cartchk, evd)
ad =getappdata(0, 'ModelViewerData');
ad = loadModels(ad);
setappdata(0, 'ModelViewerData', ad);
initgui;
%%%%%
%
%
%%%%%
function doUpdateWalkerLimits(txt, evd)
ad =getappdata(0, 'ModelViewerData');
ad = updateShapeWalkerLimits(ad);
ad = updateAppWalkerLimits(ad);
setappdata(0, 'ModelViewerData', ad);
plotModel;

%%%%%
%
%
%%%%%
function ad = updateShapeWalkerLimits(ad)
lim = str2num(get(ad.handles.shape_range, 'String'));
ad.shapewalkerlim = lim;
sh = ad.handles.shape_pc_slider
val = get(sh, 'Value');
set(sh, 'Min', -abs(lim), 'Max', abs(lim));
if abs(val)>=abs(lim)
    set(sh, 'Value', sign(val)*lim);
end
%%%%%
%
%
%%%%%
function ad = updateAppWalkerLimits(ad)
lim = str2num(get(ad.handles.app_range, 'String'));
ad.appwalkerlim = lim;
sh = ad.handles.app_pc_slider
val = get(sh, 'Value');
set(sh, 'Min', -abs(lim), 'Max', abs(lim));
if abs(val)>=abs(lim)
    set(sh, 'Value', sign(val)*lim);
end
%%%%%
%
%
%%%%%
function details = getDetails(ad)
[modelDirec, modelfilename, ext, vers] = fileparts(ad.options.pdm);
if exist(fullfile(modelDirec,'details.mat'))==2
    temp2=load(fullfile(modelDirec,'details.mat'),'-mat');
    details=temp2.details;
else
    details=[];
    detailstr='';
end
%%%%%
%
%
%%%%%
function doReloadModel(reloadModelbtn, evd)
ad =getappdata(0, 'ModelViewerData');

%p = ad.pdm.P(:,end);
%b = ad.pdm.b(end);

ad = loadModels(ad);
ad.FLIPIND = [];
setappdata(0,'ModelViewerData',ad);
initgui;

%ad =getappdata(0, 'ModelViewerData');
%ad.pdm.P = [ad.pdm.P, p];
%ad.pdm.b = [ad.pdm.b; b];
%ad.handles.shape_modes_val = [ad.handles.shape_modes_val; 0];
%setappdata(0,'ModelViewerData',ad);

%%%%%
%
%
%%%%%
function doAddRandomShapeAxis(randaxisbtn, evd)
ad =getappdata(0, 'ModelViewerData');
P = ad.pdm.pca.P;
b = ad.pdm.b;
p = P*randn(size(P,2),1);
p = p/norm(p);
ad.pdm.P = [ad.pdm.P, p];
ad.pdm.b = [b; b(1)];
ad.handles.shape_modes_val = [ad.handles.shape_modes_val; 0];
setappdata(0, 'ModelViewerData', ad);
ad = setupSliderLimits(ad);
axis_h = ad.handles.shapelevels;
doPlotModes(axis_h, ad.handles.shape_modes_val, 'shape');
%%%%%
%
%
%%%%%
function doImportMeanShape(importmeanshpbtn, evd)
ad =getappdata(0, 'ModelViewerData');
pmdir = [pwd, filesep, ad.options.PMdirectory];
[filename, pathname] = uigetfile([pmdir, filesep, '*_pm.mat'], 'Pick an PM-file');
if isequal(filename,0) || isequal(pathname,0)
    disp('User pressed cancel');
    return;
else
    meanshape = load([pathname, filesep, filename]);
    meanshape = meanshape.pts;
    disp(['User selected ', fullfile(pathname, filename)])
end
ad.pdm.Xm = meanshape;
setappdata(0, 'ModelViewerData', ad);
plotModel;
%%%%%
%
%
%%%%%
function doCalcWeights(calcweightsbtn, evd)
ad =getappdata(0, 'ModelViewerData');
%%%%%%%%
% Take care of the shape first
%%%%%%%%
if isempty(ad.pmpts)
    uiwait(msgbox('Please load a point model file first','No pm file loaded','modal'));
    return;
end
details = getDetails(ad);
original_pts = ad.pmpts;
ad.pmpts=pmalign(ad.pmpts,[details.scaling,details.rotation,details.translation],ad.pdm.Xm);
b = ad.pdm.P'*(ad.pmpts - ad.pdm.Xm);
ad.handles.shape_modes_val = b./sqrt(ad.pdm.b);
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
%%%%%%%%
% Now its time for the appearance
%%%%%%%%
original_pts = reshape(original_pts, 2, length(original_pts)/2);
if isfield(ad, 'sfam')
    if ~isempty(ad.sfam)
        meansize_im = triwarp(ad.pmimage, original_pts', ad.sfam.ompts); %warped image
        sz = ad.sfam.siz;
        Am = ad.sfam.Am;
        P = ad.sfam.P;
        b = ad.sfam.b;
        ompts = ad.sfam.ompts;
        model_b = P'*(meansize_im(:) - Am);
        ad.handles.app_modes_val = model_b./sqrt(ad.sfam.b);
        doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
        setappdata(0, 'ModelViewerData', ad);
    end
end
setappdata(0, 'ModelViewerData', ad);
plotModel;
title(ad.handles.mainAxes, ['Model - ', ad.modelledimagename], 'Interpreter', 'none');

%%%%%
%
%
%%%%%
function doLoadPointModelFile(loadpmfilebtn, evd)
ad =getappdata(0, 'ModelViewerData');
pmdirectory = ad.options.PMdirectory;

%[filename, pathname] = uigetfile({'*.mat'}, 'Pick an PM file', [pmdirectory, filesep]);
[filename, pathname] = uigetfile({'*.jpg;*.JPG';'*.jpeg;*.JPEG'}, 'Pick an JPEG-file', ['Cropped', filesep]);

if isequal(filename,0) | isequal(pathname,0)
    disp('User pressed cancel')
else
    disp(['User selected ', fullfile(pathname, filename)])
end
pmdirectory = ad.options.PMdirectory;
pmfile = getPMFile(filename, pmdirectory);
%pmfile = filename;
if isempty(pmfile)
    uiwait(msgbox('There is no PM file for your chosen image','No pm file found','modal'));
    return;
end
pts = load([pmdirectory, filesep, pmfile]);
ad.pmpts = pts.pts;
[pmdir, pmfile, ext, vers] = fileparts(pmfile);
pmfile = pmfile(1:end-3);
imname = ['Cropped', filesep, pmfile, '.jpg'];
if exist(imname)
    ad.pmimage = double(imread(imname));
    fig = figure;
    imshow(uint8(ad.pmimage));
    title(gca, ['Original - ', pmfile], 'Interpreter', 'none');
    set(fig, 'Position', [0 0 500 500]);

else
    ad.pmimage = [];
end
ad.modelledimagename = pmfile;
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function pmfilename = getPMFile(imname, D)
files = dir([D, filesep, '*_pm.mat']);
files = {files.name};
pmfilename = [];
[impath, imname, ext, vers] = fileparts(imname);
for i=1:length(files)
    pmfile = files{i};
    [pmpath, pmfile, ext, vers] = fileparts(pmfile);
    pmfile = pmfile(1:end-3);
    if strcmp(imname, pmfile)
        pmfilename = files{i};
        return;
    end
end
return;
%%%%%
%
%
%%%%%
function doSaveImage(savebtn, evd)
ad =getappdata(0, 'ModelViewerData');
if isempty(ad.currentImage)
    uiwait(msgbox('There is no image to save in this model','No image found','modal'));
    return
end
[filename, pathname] = uiputfile({'*.jpg', 'JPEG Image (*.jpg)'}, 'Save as...');
if isequal(filename,0) | isequal(pathname,0)
    disp('User pressed cancel')
else
    disp(['User selected ', fullfile(pathname, filename)])
    
    imwrite(ad.currentImage, [pathname, filesep, filename], 'JPEG');
end
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function displayModes(axis_h, modes_val, slider_h, type)
doPlotModes(axis_h, modes_val, type);
highlightMode(1, axis_h);
doSlideModes(slider_h, [], axis_h);
%%%%%
%
%
%%%%%
function doResetShape(reset, evd)
ad =getappdata(0, 'ModelViewerData');
val = ad.currentShapeMode;
ad.handles.shape_modes_val(val) = 0;
set(ad.handles.shape_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
if val<=length(ad.handles.shape_modes_val)
    highlightMode(val, ad.handles.shapelevels);
end
%%%%%
%
%
%%%%%
function doResetApp(reset, evd)
ad =getappdata(0, 'ModelViewerData');
val = ad.currentAppMode;
ad.handles.app_modes_val(val) = 0;
set(ad.handles.app_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
if val<=length(ad.handles.app_modes_val)
    highlightMode(val, ad.handles.applevels);
end
%%%%%
%
%
%%%%%
function doResetAllShape(reset, evd)
ad =getappdata(0, 'ModelViewerData');
val = ad.currentShapeMode;
ad.handles.shape_modes_val = zeros(size(ad.handles.shape_modes_val));
set(ad.handles.shape_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
if val<=length(ad.handles.shape_modes_val)
    highlightMode(val, ad.handles.shapelevels);
end
%%%%%
%
%
%%%%%
function doResetAllApp(reset, evd)
ad =getappdata(0, 'ModelViewerData');
val = ad.currentAppMode;
ad.handles.app_modes_val = zeros(size(ad.handles.app_modes_val));
set(ad.handles.app_pc_slider, 'Value', 0);
setappdata(0, 'ModelViewerData', ad);
plotModel;
doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
if val<=length(ad.handles.app_modes_val)
    highlightMode(val, ad.handles.applevels);
end
%%%%%
%
%
%%%%%
function doWalkCurrentShapePC(walkcurrbtn, evd)
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.shape_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
while get(walkcurrbtn, 'value') == 1
    ad =getappdata(0, 'ModelViewerData');
    set(slider, 'Value', val);
    val = val + direction*step;
    if val>=maxval  | val<=minval
        direction = direction*-1;
        val = val + direction*step;
    end
    ad.handles.shape_pc_slider = slider;
    setappdata(0, 'ModelViewerData', ad);
    doUpdateShapeMode(ad.handles.shape_pc_slider);
end
return;
%%%%%
%
%
%%%%%
function doWalkCurrentAppPC(walkcurrbtn, evd)
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.app_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
while get(walkcurrbtn, 'value') == 1
    ad =getappdata(0, 'ModelViewerData');
    set(slider, 'Value', val);
    val = val + direction*step;
    if val>=maxval  | val<=minval
        direction = direction*-1;
        val = val + direction*step;
    end
    ad.handles.app_pc_slider = slider;
    setappdata(0, 'ModelViewerData', ad);
    doUpdateAppMode(ad.handles.app_pc_slider);
end
return;
%%%%%
%
%
%%%%%
function doWalkAllShapePC(walkallbtn, evd)
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.shape_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
num_modes = length(ad.handles.shape_modes_val);
walksteps = [0:step:maxval, maxval:-step:minval, minval:step:0];
broken = 0;
for i=1:num_modes
    if get(walkallbtn, 'value')==0
        broken = 1;
        break
    end
    ad.currentShapeMode = i;
    setappdata(0, 'ModelViewerData', ad);
    for ii=walksteps
        if get(walkallbtn, 'value')==0
            break
        end
        ad =getappdata(0, 'ModelViewerData');
        set(slider, 'Value', ii);
        val = val + direction*step;
        if val>=maxval  | val<=minval
            direction = direction*-1;
            val = val + direction*step;
        end
        ad.handles.shape_pc_slider = slider;
        setappdata(0, 'ModelViewerData', ad);
        doUpdateShapeMode(ad.handles.shape_pc_slider);
    end
    ad.handles.shape_modes_val = zeros(size(ad.handles.shape_modes_val));
    setappdata(0, 'ModelViewerData', ad);
end
if broken==0
    set(walkallbtn, 'Value', 0);
end
return;
%%%%%
%
%
%%%%%
function doWalkAllAppPC(walkallbtn, evd)
ad =getappdata(0, 'ModelViewerData');
slider = ad.handles.app_pc_slider;
minval = get(slider, 'Min');
maxval = get(slider, 'Max');
step = get(slider, 'SliderStep');
step = step(2);
direction = 1;
val = 0;
num_modes = length(ad.handles.app_modes_val);
walksteps = [0:step:maxval, maxval:-step:minval, minval:step:0];
broken = 0;
for i=1:num_modes
    if get(walkallbtn, 'value')==0
        broken = 1;
        break
    end
    ad.currentAppMode = i;
    setappdata(0, 'ModelViewerData', ad);
    for ii=walksteps
        if get(walkallbtn, 'value')==0
            break
        end
        ad =getappdata(0, 'ModelViewerData');
        set(slider, 'Value', ii);
        val = val + direction*step;
        if val>=maxval  | val<=minval
            direction = direction*-1;
            val = val + direction*step;
        end
        ad.handles.app_pc_slider = slider;
        setappdata(0, 'ModelViewerData', ad);
        doUpdateAppMode(ad.handles.app_pc_slider);
    end
    ad.handles.app_modes_val = zeros(size(ad.handles.app_modes_val));
    setappdata(0, 'ModelViewerData', ad);
end
if broken==0
    set(walkallbtn, 'Value', 0);
end
return;
%%%%%
%
%
%%%%%
function doUpdateShapeMode(slider, evd)
ad =getappdata(0, 'ModelViewerData');
val = ad.currentShapeMode;
if isempty(val) | (val>length(ad.handles.shape_modes_val))
    return;
end
ad.handles.shape_modes_val(val) = get(slider, 'Value');
set(ad.handles.shape_pc_num_txt, 'String', num2str(val));
set(ad.handles.shape_pc_value_txt, 'String', num2str(ad.handles.shape_modes_val(val)));
setappdata(0,'ModelViewerData',ad);
plotModel;
doPlotModes(ad.handles.shapelevels, ad.handles.shape_modes_val, 'shape');
if val<=length(ad.handles.shape_modes_val)
    highlightMode(val, ad.handles.shapelevels);
end

%%%%%
%
%
%%%%%
function doUpdateAppMode(slider, evd)
ad =getappdata(0, 'ModelViewerData');
val = ad.currentAppMode;
if isempty(val) | (val>length(ad.handles.app_modes_val))
    return;
end
ad.handles.app_modes_val(val) = get(slider, 'Value');
set(ad.handles.app_pc_num_txt, 'String', num2str(val));
set(ad.handles.app_pc_value_txt, 'String', num2str(ad.handles.app_modes_val(val)));
setappdata(0,'ModelViewerData',ad);
plotModel;
doPlotModes(ad.handles.applevels, ad.handles.app_modes_val, 'app');
if val<=length(ad.handles.app_modes_val)
    highlightMode(val, ad.handles.applevels);
end
%%%%%
%
%
%%%%%
function plotModel
ad =getappdata(0, 'ModelViewerData');
drawnow;
cla(ad.handles.mainAxes);
if get(ad.handles.show_app_model_chk, 'Value')
    ad = plotAppearanceModel(ad);
    hold(ad.handles.mainAxes, 'on');
end
hold(ad.handles.mainAxes, 'on');
ad = plotEdges(ad);
ad = plotShapeModel(ad);
hold(ad.handles.mainAxes, 'off');
togglePlotView;
axis(ad.handles.mainAxes, 'equal');
axis(ad.handles.mainAxes, [ad.xrange(1)-10 ad.xrange(2)+10, ad.yrange(1)-10 ad.yrange(2)+10]);
set(ad.handles.mainAxes, 'Color', [0 0 0]);
setappdata(0, 'ModelViewerData', ad);
title(ad.handles.mainAxes, '', 'Interpreter', 'none');
drawnow;
%%%%%
%
%
%%%%%
function setShapeSliders(b)
ad =getappdata(0, 'ModelViewerData');
setappdata(0, 'ModelViewerData', ad);
%%%%%
%
%
%%%%%
function ad = plotShapeModel(ad)
if isempty(ad.pdm)
    return;
end
if get(ad.handles.show_landmarks_chk, 'Value')==0
    return;
end
pts = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val);
X = pts';
X = X(:);
% Calculate the Euclidean Distance from the formed shape to the mean.
diff = norm(ad.pdm.Xm - X(:));
set(ad.handles.eucdisttxt, 'String', num2str(diff));
if ~isempty(ad.segmentPoint)
    pts = splitPoints(pts, ad.segmentPoint, ad.offsets);
end
ad.currentShapeModel = pts;
ad = updateAxisLimits(ad, pts);
plot(ad.handles.mainAxes, pts(:,1), pts(:,2), 'o', 'MarkerFaceColor', 'y', 'MarkerEdgeColor', 'r');
%%%%%
%
%
%%%%%
function ad = plotEdges(ad)
if isempty(ad.pdm)
    return;
end
 if get(ad.handles.edges_chk, 'Value')==0
     return;
 end
pts = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val);
if ~isempty(ad.segmentPoint)
    pts = splitPoints(pts, ad.segmentPoint, ad.offsets);
end
%for i=1:size(ad.edges,1)
%    plot(ad.handles.mainAxes, [pts(ad.edges(i,1),1), pts(ad.edges(i,2), 1)], [pts(ad.edges(i,1),2), pts(ad.edges(i,2),2)], '-', 'MarkerFaceColor', 'y', 'MarkerEdgeColor', 'r');
%end

%%%%%
%
%
%%%%%
function pts = splitPoints(pts, segpoint, offsets)
segpoint = [0; segpoint; size(pts,1)];
P = {};
for i=2:length(segpoint)
    shape_pts = pts(segpoint(i-1)+1:segpoint(i),:);
    [shape_pts] = trans_rot_points(shape_pts', offsets(i-1,1:2)', offsets(i-1,3))';
    P{i-1} = shape_pts;
end
pts = [];
for i=1:length(P)
    pts = [pts; P{i}];
end
return;
%%%%%
%
%
%%%%%
function model = getCurrShapeModel(pdm, modes_val)
Xm =pdm.Xm;
P = pdm.P;
b = pdm.b;
model_b = modes_val.*sqrt(b);
X = Xm + P*model_b;
model = reshape(X, 2, length(X)/2)';

%%%%%
%
%
%%%%%
function ad = plotAppearanceModel(ad)
if ~isfield(ad, 'sfam')
    return;
end
if isempty(ad.sfam)
    return;
end
sz = ad.sfam.siz;
Am = ad.sfam.Am;
P = ad.sfam.P;
b = ad.sfam.b;
ompts = ad.sfam.ompts;
model_b = ad.handles.app_modes_val.*sqrt(b);
A = Am + P*model_b;
I = reshape(A, [sz(2) sz(1) sz(3)]);
shape_model = getCurrShapeModel(ad.pdm, ad.handles.shape_modes_val);

%%%%%
% For warping scheme w/Rico
%%%%%
% [ompts,shape_model] = tri_interp_pts(ompts,shape_model);
% [ompts,shape_model] = tri_interp_pts(ompts,shape_model);
% if isfield(ad, 'FLIPIND')
%     shape_model = collectPoints(ad.FLIPIND, shape_model);
% else
%     ad.FLIPIND = [];
% end
% 
% maxiter = 5;
% iter = 1;
% f_ind = [1];
% while ~isempty(f_ind) & (iter<=maxiter)
%     [f_ind] = accumulatePoints(ompts, shape_model);
%     ad.FLIPIND = [ad.FLIPIND, f_ind];  
%     if isfield(ad, 'FLIPIND')
%         shape_model = collectPoints(ad.FLIPIND, shape_model);
%     else
%         ad.FLIPIND = [];
%     end
%     iter = iter + 1;
% end
% iter
% %drawnow;
% %showFlipped(ompts, shape_model);
%%%%
% End of warping scheme
%%%%

if get(ad.handles.tps_chk, 'Value')
    fprintf('Calculating TPS warp...');
    %ad.currentImage = uint8(tpswarp_c(I,ompts,shape_model, 1.1));
    ad.currentImage = TPS_im_warp(I,shape_model,ompts);
    fprintf('done.\n');
else

    ad.currentImage = uint8(triwarp(I,ompts,shape_model));
end
imagesc(ad.currentImage, 'Parent', ad.handles.mainAxes);
return;
%%%%%
%
%
%%%%%
function [pts] = collectPoints(flipind, pts)
for i=1:length(flipind)
    p = flipind(i).point;
    n = flipind(i).neighbours;
    v1 = (pts(n(1),:) - pts(n(2),:))';
    v2 = (pts(p,:) - pts(n(2),:))';
    if (v1'*v1)>0
        pts(p,:) = pts(n(2),:) + ((v1'*v2)/(v1'*v1))*v1';
    end
end
return

function showFlipped(ipts, opts)
tri = delaunay(ipts(:,1), ipts(:,2),{'QJ','QJ','QJ'});
flipped_tri_ind = find_flipped_triangles(tri, ipts, opts);

figure(1); clf; hold on;
triplot(tri, opts(:,1), opts(:,2)); axis image ij;

for i=1:length(flipped_tri_ind);
    ind = flipped_tri_ind(i);
    pts = opts(tri(ind,:),:);
    p = patch(pts(:,1), pts(:,2), 'r');
end
plot(opts(:,1), opts(:,2), 'rs');
return;


%%%%%
%
%
%%%%%
function doSlideModes(slider, evd, axes_h)
ad =getappdata(0, 'ModelViewerData');
YLIM = -get(slider,'value')+sort([0 ad.dy])-.5;
set(axes_h,'ylim',YLIM);
return;
%%%%%
%
%
%%%%%
function showShapeModeInfo(mode)
ad =getappdata(0, 'ModelViewerData');
if isempty(mode) | (mode>length(ad.handles.shape_modes_val))
    return;
end
set(ad.handles.shape_pc_num_txt, 'String', num2str(mode));
set(ad.handles.shape_pc_value_txt, 'String', num2str(ad.handles.shape_modes_val(mode)));
set(ad.handles.shape_pc_slider, 'Value', ad.handles.shape_modes_val(mode));
setappdata(0,'ModelViewerData',ad);
return;
%%%%%
%
%
%%%%%
function showAppModeInfo(mode)
ad =getappdata(0, 'ModelViewerData');
if isempty(mode) | (mode>length(ad.handles.app_modes_val))
    return;
end
set(ad.handles.app_pc_num_txt, 'String', num2str(mode));
set(ad.handles.app_pc_value_txt, 'String', num2str(ad.handles.app_modes_val(mode)));
set(ad.handles.app_pc_slider, 'Value', ad.handles.app_modes_val(mode));
setappdata(0,'ModelViewerData',ad);
return;
%%%%%
%
%
%%%%%
function doSelectMode(axis_h, evd, slider_h, str)
ad =getappdata(0, 'ModelViewerData');
p = get(axis_h, 'CurrentPoint');
y = round(p(1,2));
switch str
    case 'shape'
        modes_val = ad.handles.shape_modes_val;
        ad.currentShapeMode = y;
        showShapeModeInfo(y);
        doPlotModes(axis_h, modes_val, 'shape');

    case 'app'
        modes_val = ad.handles.app_modes_val;
        ad.currentAppMode = y;
        showAppModeInfo(y);
        doPlotModes(axis_h, modes_val, 'app');
    otherwise
        disp('Unknown model type');
end
hold(axis_h, 'on');
if y<=length(modes_val)
    highlightMode(y, axis_h);
end
setappdata(0, 'ModelViewerData', ad);
return;
%%%%%
%
%
%%%%%
function doPlotModes(axis_h, modes_val, type)
ad =getappdata(0, 'ModelViewerData');
width = 6;
height = 10;
xoffset = 4;
switch type
    case 'shape'
        walkerlim = ad.shapewalkerlim;
    case 'app'
        walkerlim = ad.appwalkerlim;
end
YLIM = get(axis_h, 'YLim');
cla(axis_h);
hold(axis_h, 'on');
N = length(modes_val);
for i=1:N
    str = sprintf('%s', 'PC ', num2str(i));
    text(1, i, str,  'Color', 'k', 'HitTest', 'off', 'Clipping', 'on', 'Parent', axis_h, 'FontSize', 8);
    if modes_val(i)>0
        color = 'r';
    else
        color = 'b';
    end
    X = [xoffset xoffset+(modes_val(i)/walkerlim)];
    plot(axis_h, X,[i i], '-', 'Color', color, 'LineWidth', 4);
end
axis(axis_h, [1-.2 width .5 height+.5], 'ij');
set(axis_h, 'XMinorGrid', 'on');
set(axis_h, 'XTick', [], 'YTick', [], 'Clipping', 'on', 'Box', 'on');
set(axis_h, 'YLim', YLIM);
%%%%%
%
%
%%%%%
function highlightMode(mode, axis_h)
width = 2.5;
X = [1 width-.2 width-.2 1];
Y = [mode-.4 mode-.4 mode+.4 mode+.4];
C = [0.1412    0.3137    1];
%patch(X, Y, C, 'FaceAlpha', .1, 'Parent', axis_h, 'EdgeColor', 'k', 'MarkerSize', .1, 'LineWidth', 0.01, 'LineStyle', ':', 'Clipping', 'on');
patch(X, Y, C, 'Parent', axis_h, 'Clipping', 'on');
str = sprintf('%s', 'PC ', num2str(mode));
text(1, mode, str, 'Clipping', 'on', 'Parent', axis_h, 'FontSize', 8, 'HitTest', 'off', 'Color', 'w');
return;
%%%%%%%%%%%%%%%%%%%%
%
% Disable Appearance buttons
%
%%%%%%%%%%%%%%%%%%%%
function handles = toggleAppearanceGUI(handles, enablestr)
set(handles.applevels, 'Visible', enablestr);
set(handles.appslider, 'Enable', enablestr);
set(handles.app_pc_slider, 'Enable', enablestr);
set(handles.walk_curr_app_pc_btn, 'Enable', enablestr);
set(handles.walk_all_app_pc_btn, 'Enable', enablestr);
set(handles.reset_app_btn, 'Enable', enablestr);
set(handles.reset_all_app_btn, 'Enable', enablestr);
set(handles.app_range, 'Enable', enablestr);


%%%%%%%%%%%%%%%%%%%%
%
% SetGUILevel
%
%%%%%%%%%%%%%%%%%%%%
function handles = SetGUILevel(handles, level)
if level==1
    set(handles.gen_opts_panel, 'Visible', 'off');
    set(handles.add_rand_axis_btn, 'Visible', 'off');
    set(handles.reload_model_btn, 'Visible', 'off'); 
end