/****************************************************************************************
 *
 * Author:		Barry Theobald, Iain Matthews
 * Data:			30/12/2002
 * Function:	Use windows API calls to set up an offscreen bitmap for image warping.
 *
 * Revision:	$Id: gltriwarp.c,v 1.0 2003-12-08 11:32:56+00 barry-john_theobald Exp barry-john_theobald $
 * Revision:	$Id: gltriwarp.c,v 1.0 2005-09-26 15:34:00+00 andrew ian hanna Exp andrew ian hanna $
 * Revision:	$Id: gltriwarp.c,v 1.0 2006-02-07 16:08:00+00 andrew (god) courtenay Exp andrew (god) courtenay $

****************************************************************************************/
#include <stdio.h>
#include <math.h>
#include <GL/glut.h>

#include "GLSL\GL_Ext.h"
#include "GLSL\GL_ShaderOBJ.h"
#include "GLSL\GL_FrameBufferObject.h"

#include "mex.h"

// Define globals **********************************************************************

#define TEXMAP_0            0
#define TEXMAP_1            1
#define BMP_WIDTH           1024
#define BMP_HEIGHT          1024

static GL_Ext				glextensions;					// OpenGL extension object
static GL_ShaderOBJ         shaderobj;						// OpenGL Shader object
static GL_FrameBufferObject fbobject;                      // OpenGL Framebuffer object

static unsigned char*       image_in;
static int                  image_width;
static int                  image_height;
static int                  image_numbchannels;
mxArray*                    image_out;

// Function declarations ****************************************************************
void initGL(void);
void initShader(void);
void bindTextureGL(unsigned char* data, unsigned int w, unsigned int h, unsigned int bpp, const int TexId);
void saveFramebufferGL(unsigned char* data, unsigned int w, unsigned int h, unsigned int bpp);
void displayGL(void);

void getParsedData(int nrhs, const mxArray *prhs[]);
GLubyte *GetInputImage(const mxArray *img, int *width, int *height, int *numbchannels);

// **************************************************************************************
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
    //get the parsed data into a suitable form
    getParsedData(nrhs, prhs);

    //Initialise Glut
    glutCreateWindow("MEX GLUT");  

    //Initialise OpenGL states
    initGL();
    
    //Do our rendering
    displayGL();
      
    //Set output image
	plhs[0] = image_out;
    
    //Clean up...
    if(image_in) delete[] image_in;
    glutDestroyWindow(glutGetWindow());
    //mxFree(image_out);
}
void displayGL(void)
{
    glViewport(0, 0, image_width, image_height);
    glMatrixMode(GL_PROJECTION);    
    glLoadIdentity();               
    gluOrtho2D(-1, 1, -1, 1);       
    glMatrixMode(GL_MODELVIEW);     
    glLoadIdentity();    
    
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT ); 
    glLoadIdentity();               

    fbobject.bind();
	fbobject.switchTarget(TEXMAP_1);

	shaderobj.enable();
	glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, TEXMAP_0);

  	glColor3f(1.0, 1.0, 1.0);
    glBegin(GL_QUADS);
            glTexCoord2f(0, 0); glVertex3f(-1, -1, -0.5f);
            glTexCoord2f(1, 0); glVertex3f( 1, -1, -0.5f);
            glTexCoord2f(1, 1); glVertex3f( 1,  1, -0.5f);
            glTexCoord2f(0, 1); glVertex3f(-1,  1, -0.5f);
    glEnd();

	glDisable(GL_TEXTURE_2D);
	shaderobj.disable();
    
	//read from front buffer pixel data to our mxarray
	glReadBuffer(GL_FRONT);
	glReadPixels(0, 0, image_width, image_height, GL_RGB, GL_UNSIGNED_BYTE, (GLubyte*)mxGetData(image_out));

    fbobject.stop();
}

void bindTextureGL(unsigned char* data, unsigned int w, unsigned int h, unsigned int bpp, const int TexId)
{
 	GLenum   format  =  (bpp==4) ? GL_RGBA : GL_RGB;
    
    glBindTexture (GL_TEXTURE_2D, TexId); 
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
	glTexImage2D (GL_TEXTURE_2D, 0, format, w, h, 0, GL_RGB, GL_UNSIGNED_BYTE, data);
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
 
}

void initShader(void)
{
    GLcharARB *VertexShaderSource, *FragmentShaderSource;
    shaderobj.read_shaderSRC("Shaders/sobel", &VertexShaderSource, &FragmentShaderSource);
	shaderobj.install_shader(VertexShaderSource, FragmentShaderSource);

	shaderobj.enable();
	shaderobj.sendUniform1f("texwidth", BMP_WIDTH);
	shaderobj.disable();
}

void initGL(void)
{
    glextensions.load_GLextension();
	fbobject.glext = shaderobj.glext = &glextensions;
	shaderobj.disable();
    
 	bindTextureGL(image_in, image_width, image_height, image_numbchannels, TEXMAP_0);
	bindTextureGL(NULL, image_width, image_height, image_numbchannels, TEXMAP_1);

    fbobject.initialize(image_width, image_height, GL_RGBA);
	initShader();
}

//********************************  Matlab stuff   ***************************************

void getParsedData(int nrhs, const mxArray *prhs[])
{
    //Read in source image
    image_in = GetInputImage(prhs[0], &image_width, &image_height, &image_numbchannels);

    //Allocate memory for output image
	int dims[3];
 	dims[0] = image_width;
	dims[1] = image_height;
	dims[2] = image_numbchannels;
    image_out = mxCreateNumericArray(3, dims, mxUINT8_CLASS, mxREAL);
	if ((image_out) == NULL) mexErrMsgTxt("Unable to allocate output matrix\n");
}
//****************************************************************************************
//****************************************************************************************
//********************************  Iain's Stuff   ***************************************
//****************************************************************************************
//****************************************************************************************

GLubyte *GetInputImage(const mxArray *img, int *width, int *height, int *numbchannels)
{
	int	ndim, nx, ny;
	const int *dim;

    if(!mxIsNumeric(img) || mxIsEmpty(img) || !mxIsUint8(img))
		mexErrMsgTxt("Input image must be uint8 2D or 3D numeric matrix");
		
	ndim = mxGetNumberOfDimensions(img);
	if(ndim != 3)
		mexErrMsgTxt("Input image must be (nx * ny * 3) 24-bit RGB");

	dim = mxGetDimensions(img);
	if(dim[2] > 3)
		mexWarnMsgTxt("Only using first 3 values of 3rd dimension");
	else if(dim[2] < 3)
		mexErrMsgTxt("Third dimension must have size = 3");

    *width = nx = dim[0];
	*height = ny = dim[1];
    *numbchannels = dim[2];

	return((GLubyte*)mxGetData(img));
}

